/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
neck_score();

/**
 * Performs the neck score calculation for the specified models
 */
function neck_score() {
    let template = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = "neck";

    let occupants = ["DRIVER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Get the head excursion zone and whether there's a countermeasure or not - required to set the high score value */
            let head_excursion_zone = get_variable_value(status, `${m}_STRUCTURE_HEAD_EXCURSION_RATING`, "string");
            let countermeasure_int = get_variable_value(
                status,
                `${m}_STRUCTURE_HEAD_EXCURSION_COUNTERMEASURE_VALUE`,
                "int"
            );

            /* comment why checking for countermeasure_int === 1 */
            let countermeasure = countermeasure_int === 1;

            let NECK_HI_SCORE = get_max_neck_score(head_excursion_zone, countermeasure);

            new Variable(
                template,
                `NECK_HI_SCORE`,
                `Max neck score for head excursion in ${head_excursion_zone} zone with${
                    countermeasure ? "" : "out"
                } countermeasure.`,
                NECK_HI_SCORE.toString(),
                "General",
                false,
                true
            );

            /* Scale scores from T/HIS to the allowable hi score */
            let NECK_UPPER_TENSION_HI_LIMIT = 3.74; //kN
            let NECK_UPPER_TENSION_LO_LIMIT = 3.74; //kN
            let NECK_UPPER_LATERAL_FLEXION_HI_LIMIT = 162; //Nm
            let NECK_UPPER_LATERAL_FLEXION_LO_LIMIT = 248; //Nm
            let NECK_UPPER_NEG_EXTENSION_HI_LIMIT = 50; //Nm
            let NECK_UPPER_NEG_EXTENSION_LO_LIMIT = 50; //Nm
            let max_upper_tension = get_variable_value(status, `${m}_${occ}_NECK_UPPER_AXIAL_MAX_VALUE`, "float");
            let max_upper_flexion = get_variable_value(status, `${m}_${occ}_NECK_UPPER_FLEXION_MAX_VALUE`, "float");
            let max_upper_extension = get_variable_value(status, `${m}_${occ}_NECK_UPPER_EXTENSION_MAX_VALUE`, "float");

            if (max_upper_tension == "Missing" || max_upper_tension == null) {
                var upper_tension_score = 0;
            } else {
                var upper_tension_score = sliding_scale(
                    max_upper_tension,
                    NECK_UPPER_TENSION_HI_LIMIT,
                    NECK_UPPER_TENSION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            if (max_upper_flexion == "Missing" || max_upper_flexion == null) {
                var upper_flexion_score = 0;
            } else {
                var upper_flexion_score = sliding_scale(
                    max_upper_flexion,
                    NECK_UPPER_LATERAL_FLEXION_HI_LIMIT,
                    NECK_UPPER_LATERAL_FLEXION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            if (max_upper_extension == "Missing" || max_upper_extension == null) {
                var upper_extension_score = 0;
            } else {
                var upper_extension_score = sliding_scale(
                    max_upper_extension,
                    NECK_UPPER_NEG_EXTENSION_HI_LIMIT,
                    NECK_UPPER_NEG_EXTENSION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            new Variable(
                template,
                `${m}_${occ}_NECK_UPPER_AXIAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                upper_tension_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_NECK_UPPER_FLEXION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                upper_flexion_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_NECK_UPPER_EXTENSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                upper_extension_score.toFixed(3),
                "General",
                false,
                true
            );
            let un_score = Math.min(upper_tension_score, upper_flexion_score, upper_extension_score);
            new Variable(
                template,
                `${m}_${occ}_NECK_UPPER_SCORE`,
                `Result from Automotive Assessments Workflow`,
                un_score.toFixed(3),
                "General",
                false,
                true
            );

            let NECK_LOWER_TENSION_HI_LIMIT = 3.74; //kN
            let NECK_LOWER_TENSION_LO_LIMIT = 3.74; //kN
            let NECK_LOWER_LATERAL_FLEXION_HI_LIMIT = 162; //Nm
            let NECK_LOWER_LATERAL_FLEXION_LO_LIMIT = 248; //Nm
            let NECK_LOWER_NEG_EXTENSION_HI_LIMIT = 100; //Nm Note: only monitored in 2020-2022 (not used for scoring)
            let NECK_LOWER_NEG_EXTENSION_LO_LIMIT = 100; //Nm Note: only monitored in 2020-2022 (not used for scoring)
            let max_lower_tension = get_variable_value(status, `${m}_${occ}_NECK_LOWER_AXIAL_MAX_VALUE`, "float");
            let max_lower_flexion = get_variable_value(status, `${m}_${occ}_NECK_LOWER_FLEXION_MAX_VALUE`, "float");
            let max_lower_extension = get_variable_value(
                status,
                `${m}_${occ}_NECK_LOWER_EXTENSION_ABS_MIN_VALUE`,
                "float"
            );

            if (max_lower_tension == "Missing" || max_lower_tension == null) {
                var lower_tension_score = 0;
            } else {
                var lower_tension_score = sliding_scale(
                    max_lower_tension,
                    NECK_LOWER_TENSION_HI_LIMIT,
                    NECK_LOWER_TENSION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            if (max_lower_flexion == "Missing" || max_lower_flexion == null) {
                var lower_flexion_score = 0;
            } else {
                var lower_flexion_score = sliding_scale(
                    max_lower_flexion,
                    NECK_LOWER_LATERAL_FLEXION_HI_LIMIT,
                    NECK_LOWER_LATERAL_FLEXION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            if (max_lower_extension == "Missing" || max_lower_extension == null) {
                var lower_extension_score = 0;
            } else {
                var lower_extension_score = sliding_scale(
                    max_lower_extension,
                    NECK_LOWER_NEG_EXTENSION_HI_LIMIT,
                    NECK_LOWER_NEG_EXTENSION_LO_LIMIT,
                    NECK_HI_SCORE,
                    0
                );
            }

            new Variable(
                template,
                `${m}_${occ}_NECK_LOWER_AXIAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                lower_tension_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_NECK_LOWER_FLEXION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                lower_flexion_score.toFixed(3),
                "General",
                false,
                true
            );
            if (lower_extension_score > 0) {
                var lower_extension_monitor = "Pass";
            } else {
                var lower_extension_monitor = "Fail";
            }
            new Variable(
                template,
                `${m}_${occ}_NECK_LOWER_EXTENSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                lower_extension_monitor,
                "General",
                false,
                true
            );
            let ln_score = Math.min(lower_tension_score, lower_flexion_score);
            new Variable(
                template,
                `${m}_${occ}_NECK_LOWER_SCORE`,
                `Result from Automotive Assessments Workflow`,
                ln_score.toFixed(3),
                "General",
                false,
                true
            );

            let neck_score = Math.min(un_score, ln_score);
            new Variable(
                template,
                "NECK_FINAL_SCORE",
                "Result from Automotive Assessments Workflow",
                neck_score.toString(),
                "General",
                false,
                true
            );
        }
    }
}

/**
 * Returns the maximum neck score for the given head excursion zone and countermeasure
 *
 * @param {string} zone The head excursion zone
 * @param {boolean} countermeasure Whether there's a countermeasure or not
 * @returns {number} The maximum neck score
 */
function get_max_neck_score(zone, countermeasure) {
    if (countermeasure) {
        switch (zone) {
            case "CAPPING":
            case "RED":
                return 4;
            case "BROWN":
                return 4;
            case "ORANGE":
                return 3;
            case "YELLOW":
                return 4;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    } else {
        switch (zone) {
            case "CAPPING":
                return 0;
            case "RED":
                return 1;
            case "ORANGE":
                return 1;
            case "YELLOW":
                return 2;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
